<?php
/* --------------------------------------------------------------
 AdminController.php  2019-11-28
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Application\Http;

use Exception;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Admin\Layout\Smarty\LayoutLoader;
use Gambio\Admin\Layout\Smarty\TemplateTranslations;
use Gambio\Core\Language\TextManager;
use Smarty;

/**
 * Class AdminController
 * @package Gambio\Core\Application\Http
 */
abstract class AdminController
{
    /**
     * @var LayoutLoader
     */
    private $layoutLoader;
    
    /**
     * @var Smarty
     */
    private $smarty;
    
    /**
     * @var TextManager
     */
    private $textManager;
    
    /**
     * @var ContentNavigation
     */
    private $contentNavigation;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var TemplateTranslations
     */
    private $templateTranslations;
    
    
    /**
     * Admin controller initialization.
     *
     * @param LayoutLoader         $layoutLoader
     * @param Smarty               $smarty
     * @param TextManager          $textManager
     * @param ContentNavigation    $contentNavigation
     * @param UserPreferences      $userPreferences
     * @param TemplateTranslations $templateTranslations
     */
    public function initialize(
        LayoutLoader $layoutLoader,
        Smarty $smarty,
        TextManager $textManager,
        ContentNavigation $contentNavigation,
        UserPreferences $userPreferences,
        TemplateTranslations $templateTranslations
    ): void {
        $this->layoutLoader         = $layoutLoader;
        $this->smarty               = $smarty;
        $this->textManager          = $textManager;
        $this->contentNavigation    = $contentNavigation;
        $this->userPreferences      = $userPreferences;
        $this->templateTranslations = $templateTranslations;
    }
    
    
    /**
     * Renders the template.
     * Before rendering the template, all layout data will be loaded.
     *
     * @param string $pageTitle
     * @param string $template
     *
     * @return string
     */
    protected function render(string $pageTitle, string $template): string
    {
        $this->layoutLoader->load($this->smarty);
        
        $this->assign('page_title', $pageTitle);
        $this->assign('content_navigation', $this->contentNavigation->toArray());
        
        try {
            return $this->smarty->fetch('extends:' . $template);
        } catch (Exception $e) {
            return '';
        }
    }
    
    
    /**
     * Translates the given phrase.
     * The currently active language will be used for translation.
     *
     * @param string $phrase
     * @param string $section
     *
     * @return string
     */
    protected function translate(string $phrase, string $section): string
    {
        return $this->textManager->getPhraseText($phrase, $section, $this->userPreferences->languageId());
    }
    
    
    /**
     * Add a section to the template translations that will be assign to Smarty and used by the JSEngine.
     *
     * @param string $section
     */
    protected function addTemplateTranslations(string $section): void
    {
        $this->templateTranslations->addSection($section);
    }
    
    
    /**
     * Adds a template directory to Smarty.
     *
     * @param string $directory
     */
    protected function addTemplateDirectory(string $directory): void
    {
        $this->smarty->addTemplateDir($directory);
    }
    
    
    /**
     * Assign a new content navigation item.
     *
     * @param string $name
     * @param string $url
     * @param bool   $current
     */
    protected function assignContentNavItem(string $name, string $url, bool $current): void
    {
        $this->contentNavigation->add($name, $url, $current);
    }
    
    
    /**
     * Assigns a new template variable or override an existing one.
     *
     * @param string $key
     * @param mixed  $value
     */
    protected function assign(string $key, $value): void
    {
        $this->smarty->assign($key, $value);
    }
}